using System;
using System.Collections.Generic;

class Program
{
   
    public class FileSystemObject
    {
        public string Name { get; set; }
        public FileSystemObject Parent { get; set; }
        public List<FileSystemObject> Children { get; set; } = new List<FileSystemObject>();
        public bool IsDirectory { get; set; }

        public FileSystemObject(string name, bool isDirectory)
        {
            Name = name;
            IsDirectory = isDirectory;
        }

        public void MoveTo(FileSystemObject newParent)
        {
            if (Parent != null)
            {
                Parent.Children.Remove(this);
            }
            Parent = newParent;
            newParent.Children.Add(this);
        }

        public void Show()
        {
            if (IsDirectory)
            {
                Console.WriteLine("/" + Name);
                foreach (var child in Children)
                {
                    Console.WriteLine(" /" + child.Name);
                }
            }
            else
            {
                Console.WriteLine("~" + Name);
            }
        }

        public void Delete()
        {
            if (IsDirectory && Children.Count > 0)
            {
                Console.WriteLine($"Vai tiešām vēlies dzēst mapi {Name} un tās bērnus? (y/n)");
                var response = Console.ReadLine();
                if (response?.ToLower() == "y")
                {
                    foreach (var child in Children)
                    {
                        child.Delete();
                    }
                    Parent?.Children.Remove(this);
                }
            }
            else
            {
                Parent?.Children.Remove(this);
            }
        }

        public void AddChild(FileSystemObject child)
        {
            if (!Children.Exists(c => c.Name == child.Name))
            {
                Children.Add(child);
                child.Parent = this;
            }
            else
            {
                Console.WriteLine($"Objekts ar nosaukumu {child.Name} jau eksistē.");
            }
        }
    }


    class FileSystem
    {
        public FileSystemObject Root { get; set; }
        public FileSystemObject CurrentDirectory { get; set; }

        public FileSystem()
        {
            Root = new FileSystemObject("c:", true);
            CurrentDirectory = Root;
        }

        public void CreateDirectory(string name)
        {
            var directory = new FileSystemObject(name, true);
            CurrentDirectory.AddChild(directory);
        }

        public void CreateFile(string name)
        {
            var file = new FileSystemObject(name, false);
            CurrentDirectory.AddChild(file);
        }

        public void ChangeDirectory(string name)
        {
            var target = CurrentDirectory.Children.Find(c => c.Name == name && c.IsDirectory);
            if (target != null)
            {
                CurrentDirectory = target;
            }
            else
            {
                Console.WriteLine("Norādītā mape neeksistē.");
            }
        }

        public void ShowCurrentDirectory()
        {
            Console.WriteLine($"Atrašanās vieta: {CurrentDirectory.Name}");
            foreach (var child in CurrentDirectory.Children)
            {
                child.Show();
            }
        }
    }

    static void Main(string[] args)
    {
        FileSystem fileSystem = new FileSystem();

        while (true)
        {
            fileSystem.ShowCurrentDirectory();
            Console.Write("> ");
            string command = Console.ReadLine()?.Trim().ToLower();

            if (command == "exit" || command == "by")
            {
                break;
            }

            string[] commandParts = command?.Split(' ');
            switch (commandParts[0])
            {
                case "mkdir":
                    if (commandParts.Length > 1)
                    {
                        fileSystem.CreateDirectory(commandParts[1]);
                    }
                    break;

                case "create":
                    if (commandParts.Length > 1)
                    {
                        fileSystem.CreateFile(commandParts[1]);
                    }
                    break;

                case "rm":
                    if (commandParts.Length > 1)
                    {
                        var objectToDelete = fileSystem.CurrentDirectory.Children.Find(c => c.Name == commandParts[1]);
                        objectToDelete?.Delete();
                    }
                    break;

                case "dir":
                    fileSystem.ShowCurrentDirectory();
                    break;

                case "cd":
                    if (commandParts.Length > 1)
                    {
                        if (commandParts[1] == "..")
                        {
                            if (fileSystem.CurrentDirectory.Parent != null)
                            {
                                fileSystem.CurrentDirectory = fileSystem.CurrentDirectory.Parent;
                            }
                        }
                        else
                        {
                            fileSystem.ChangeDirectory(commandParts[1]);
                        }
                    }
                    break;

                case "edit":
                    if (commandParts.Length > 1)
                    {
                        var fileToEdit = fileSystem.CurrentDirectory.Children.Find(c => c.Name == commandParts[1] && !c.IsDirectory);
                        if (fileToEdit != null)
                        {
                            Console.WriteLine($"Rediģēt datni: {fileToEdit.Name}");
                        }
                        else
                        {
                            Console.WriteLine("Norādītā datne neeksistē.");
                        }
                    }
                    break;

                default:
                    Console.WriteLine("Nerakstīta komanda.");
                    break;
            }
        }
    }
}